#!/bin/python3
'''
    SPDX-FileCopyrightText: 2025 Agata Cacko <cacko.azh@gmail.com>

    This file is part of Fast Sketch Cleanup Plugin for Krita

    SPDX-License-Identifier: GPL-3.0-or-later
'''


from PyQt5.QtWidgets import QDialog, QHBoxLayout, QVBoxLayout, QPushButton, QLabel
from PyQt5.QtGui import QPixmap
from PyQt5.QtCore import QObject, Qt, pyqtSignal, QEvent

class PreviewImagesViewerEventFilter(QObject):

	keyPressedCaught = pyqtSignal(QEvent)

	def eventFilter(self, object, event):
		if event.type() == QEvent.KeyPress:
			if event.key() == Qt.Key_Left or event.key() == Qt.Key_Right:
				self.keyPressedCaught.emit(event)
				return True

		return False


class PreviewImagesViewer(QDialog):

	mainPictureLabel : QLabel = None

	images = None
	currentIndex = 0

	eventFilter = None



	def __init__(self, images, whichImage = 0, parent = None):
		super().__init__(parent)
		self.images = images


		mainLayout = QVBoxLayout()
		self.setLayout(mainLayout)
		self.mainPictureLabel = QLabel()
		self.mainPictureLabel.setMinimumSize(100, 100)
		self.mainPictureLabel.setAlignment(Qt.AlignCenter)

		mainLayout.addWidget(self.mainPictureLabel)

		self.titleLabel = QLabel("Title")
		mainLayout.addWidget(self.titleLabel)
		self.titleLabel.setAlignment(Qt.AlignCenter)
		self.titleLabel.setFixedHeight(40)
		

		buttonsBelowLayout = QHBoxLayout()
		mainLayout.addLayout(buttonsBelowLayout)

		leftButton = QPushButton("<")
		rightButton = QPushButton(">")
		
		buttonsBelowLayout.addWidget(leftButton)
		buttonsBelowLayout.addWidget(rightButton)
		
		
		leftButton.pressed.connect(self.leftButtonPressed)
		rightButton.pressed.connect(self.rightButtonPressed)

		self.titles = ["1. Original picture", "2. After pre-processing", "3. After the plugin", "4. After post-processing"]
		self.setPictureIndex(self.currentIndex)

		self.eventFilter = PreviewImagesViewerEventFilter()
		leftButton.installEventFilter(self.eventFilter)
		rightButton.installEventFilter(self.eventFilter)

		self.eventFilter.keyPressedCaught.connect(self.keyPressEvent)

		self.resize(1000, 1000)

		self.setPictureIndex(whichImage)

	
	def leftButtonPressed(self):
		self.changeThePicture(False)

	def rightButtonPressed(self):
		self.changeThePicture(True)

	def setPictureIndex(self, index: int):
		index = max(min(len(self.images) - 1, index), 0)
		self.currentIndex = index
		image = self.images[index]
		w = self.mainPictureLabel.width()
		h = self.mainPictureLabel.height()
		scaled = image
		if w < image.width() or h < image.height():
			scaled = image.scaled(w, h, Qt.KeepAspectRatio, Qt.FastTransformation)
		else:
			scaled = image.scaled(w, h, Qt.KeepAspectRatio)
		self.mainPictureLabel.setPixmap(QPixmap(scaled))

		if index >= 0 and index < len(self.titles):
			self.titleLabel.setText(self.titles[index])


	def changeThePicture(self, next: bool):
		if len(self.images) == 0:
			return
		
		if next:
			self.currentIndex += 1
			if (self.currentIndex >= len(self.images)):
				self.currentIndex -= len(self.images)
		else:
			self.currentIndex -= 1

			if (self.currentIndex < 0):
				self.currentIndex += len(self.images)
		
		self.setPictureIndex(self.currentIndex)


	def resizeEvent(self, a0):
		self.setPictureIndex(self.currentIndex)
		return super().resizeEvent(a0)
	

	def keyPressEvent(self, event) -> None:
		if event.key() == Qt.Key_Left:
			self.changeThePicture(False)
			return
		elif event.key() == Qt.Key_Right:
			self.changeThePicture(True)
			return
		
		super().keyPressEvent(event)

